/*
 * TrackerThread.java
 *
 */

package notorrent.tracker;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.Runnable;
import java.net.ServerSocket;
import java.net.Socket;
import notorrent.messages.Message;
import notorrent.messages.MessageError;
import notorrent.util.Debug;
import notorrent.util.UncheckedException;
import org.jdom.input.JDOMParseException;

/**
 * Waits for a client connection; handles that client connection; and
 * then waits for another connection.
 *
 * @author Howie
 */
public class TrackerThread implements Runnable
{
    /** to keep track of the separate handler threads */
    private static int nextThreadId	= 0;
    private int threadId		= -1;
    
    private ServerSocket serverSocket	= null;
    
    /**
     * Creates a new instance of TrackerThread
     */
    public TrackerThread(ServerSocket serverSocket)
    {
        this.threadId = nextThreadId++;
        this.serverSocket = serverSocket;
    }
    
    public void run()
    {
        Socket clientSocket	    = null;
        
        // run forever, accepting and handling each connection
        while (true)
        {
            try
            {
                StateInfo.showTrackedResources();
                
                printStatus("waiting for connection from a client.");
                clientSocket = serverSocket.accept();  // block waiting for connection
                
                String clientName = clientSocket.getInetAddress().getHostAddress() + ":" + clientSocket.getPort();
                printStatus("accepted connection from a client (" + clientName + ").  going to handle client's message now.");
                handleRequest(clientSocket);
                
                clientSocket.close();
                
                printStatus("closed connection with client (" + clientName + ").");
            }
            catch (IOException e)
            {
                e.printStackTrace();
            }
        }
    }
    
    private void handleRequest(Socket clientSocket)
    {
        try
        {
            Message requestMessage = readClientsMessage(clientSocket);
            System.out.println("received message from client: " + requestMessage);
            
            Message responseMessage = requestMessage.handle();
            
            printStatus("received and handled a message.");
            
            if (responseMessage != null)
            {
                BufferedOutputStream outputStream = new BufferedOutputStream(clientSocket.getOutputStream());
                
                System.out.println("going to send response message: " + responseMessage);
                
                outputStream.write(responseMessage.encode());
                outputStream.flush();
                
                printStatus("sent response message to client.");
            }
        }
        catch (UncheckedException e)
        {
            if (e.originalException instanceof JDOMParseException)
            {
                Debug.printDebug("Error parsing client's request.  Perhaps the connection terminated abnormally.");
            }
            else
            {
                Debug.printDebug(e);
            }
        }
        catch (Exception e)
        {
            Debug.printDebug(e);
            sendErrorMessageToClient(clientSocket, e);
        }
    }
    
    private void sendErrorMessageToClient(Socket clientSocket, Exception exception)
    {
        try
        {
            BufferedOutputStream outputStream = new BufferedOutputStream(clientSocket.getOutputStream());
            byte[] errorMessage = (new MessageError("Error on tracker decoding or handling message received from client (you).\n" + exception.toString())).encode();
            
            outputStream.write(errorMessage);
            outputStream.flush();
            
            printStatus("sent error message to client");
        }
        catch (IOException e)
        {
            Debug.printDebug("error sending error message to client.");
            Debug.printDebug(e);
        }
    }
    
    private Message readClientsMessage(Socket clientSocket)
    {
        try
        {
            InputStream inputStream = clientSocket.getInputStream();
            
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            
            int BUFSIZE = 32;
            int recvMsgSize;
            byte[] byteBuffer = new byte[BUFSIZE];
            
            while ((recvMsgSize = inputStream.read(byteBuffer)) != -1)
            {
                baos.write(byteBuffer, 0,  recvMsgSize);
            }
            
            return Message.decode(baos.toByteArray());
        }
        catch (Exception e)
        {
            printStatus("Error decoding or handling client's message.");
            throw new UncheckedException(e);
        }
    }
    
    private void printStatus(String message)
    {
        System.out.println("HandlerThread #" + threadId + ": " + message);
    }
}
