/*
 * StateInfo.java
 *
 */

package notorrent.tracker;

import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import notorrent.util.MethodNotImplementedException;
import notorrent.util.UncheckedException;

/**
 * singleton object containing state info, specifically info on
 * which peers exists, and
 * which peers hold which resources.
 *
 * @author Howie
 */
public class StateInfo
{
    private static StateInfo stateInfo			= null;
    
    /** represents the number of time windows that have elapsed */
    private int time					= 0;
    
    /**
     * returns the current time in terms of number of time windows that have elapsed
     */
    public static int currentTime()
    {
        return stateInfo.time;
    }
    
    private int numTimeWindows;
    
    /**
     * the number of time windows to keep track of.
     * i.e. if a time window is an hour long and numTimeWindows is 3,
     * we'll keep track of the number of requests in each of the last 3 hours.
     */
    public static int getNumTimeWindows()
    {
        return stateInfo.numTimeWindows;
    }
    
    /**
     * the resources that we're tracking.
     * a ResourceInfo object maps a URI to a list of peers that claim to have a
     * copy of that resource.
     */
    private Hashtable<URI, ResourceInfo> resources		= null;
    
    /** Creates a new instance of StateInfo */
    public StateInfo(CommandLineArgs commandLineArgs)
    {
        this.resources	= new Hashtable<URI, ResourceInfo>();
        
        this.numTimeWindows = commandLineArgs.getNumTimeWindows();
    }
    
    public static void initialize(CommandLineArgs commandLineArgs)
    {
        stateInfo = new StateInfo(commandLineArgs);
        
//	try
//	{
//	    // add some dummy data for now
//	    Peer aPeerWithFoo = new Peer(InetAddress.getByName("1.2.3.4"), 5678);
//	    URI resource = new URI("http://foo/");
//	    StateInfo.iHaveResource(resource, aPeerWithFoo);
//	}
//	catch (UnknownHostException e)
//	{
//	    throw new UncheckedException(e);
//	}
//	catch (URISyntaxException e)
//	{
//	    throw new UncheckedException(e);
//	}
    }
    
    /**
     * returns a Set of Peers that claim to have the given resource.
     * also updates the request counter for the given resource.
     * if the resource is not in the resources table, and empty Set
     * is returned.
     */
    public static synchronized Set<Peer> getPeersWithResource(URI requestedResource)
    {
        ResourceInfo resourceInfo = stateInfo.resources.get(requestedResource);
        
        // keep track of # of requests per time unit by indicating that a request
        // for this resource is being made right now.
        if (resourceInfo != null)
        {
            resourceInfo.updateRequestCounter();
        }
        
        // if no peers have the given resource, return an empty set (instead of a null)
        return resourceInfo == null ? new HashSet<Peer>() : resourceInfo.getPeersWithResource();
    }
    
    /**
     * stores the given peer and resource in the resources table.
     * if the resource was already in the resources table, this peer will be added
     * to that entry.
     * if the resource was not in the resources table, a new ResourceInfo object will be
     * created, the peer will be added to that ResourceInfo object, and the resource will
     * be added to the resources table.
     */
    public static synchronized void iHaveResource(URI resource, Peer me)
    {
        ResourceInfo resourceInfo = stateInfo.resources.get(resource);
        
        // if no ResourceInfo object exists for the resource, create one
        // and put it in the resources table.
        if (resourceInfo == null)
        {
            resourceInfo =  new ResourceInfo();
            stateInfo.resources.put(resource, resourceInfo);
        }
        
        // add peer to Set of Peers that claim to have resource
        resourceInfo.addPeer(me);
    }
    
    public static synchronized void peerFailedToServe(URI resource, Peer peerThatClientClaimedFailed)
    {
        if (peerIsDown(peerThatClientClaimedFailed)) // check if the peer is actually down.
        {
            evictPeer(peerThatClientClaimedFailed);
        }
    }
    
    private static synchronized boolean peerIsDown(Peer peer)
    {
        // we trust client that peer actually failed "for now."
        // we should actually ping the peer to check if it is down.
        return true;
//        throw new MethodNotImplementedException();
    }
    
    /**
     * evict peer from each ResourceInfo object in tracker's resources table
     */
    private static synchronized void evictPeer(Peer peer)
    {
        for (ResourceInfo resourceInfoObject : stateInfo.resources.values())
        {
            resourceInfoObject.evictPeer(peer);
        }
    }
        
    /**
     * show which resources are being tracked and which peers claim to have them.
     * output will look like this:
     * <pre>Tracked Resources:
     * ------------------
     * http://www.foo.com:
     * 1.2.3.4:5678
     * 2.3.4.5:3456
     * ------------------
     * http://bar.com:
     * 3.5.6.7:4283
     * ------------------</pre>
     */
    public static synchronized void showTrackedResources()
    {
        String separator = "------------------";
        
        System.out.println("Tracked Resources");
        System.out.println(separator);
        
        Set<Map.Entry<URI, ResourceInfo>> resourceSet = stateInfo.resources.entrySet();
        for (Map.Entry<URI, ResourceInfo> resource : resourceSet)
        {
            // print uri
            System.out.println(resource.getKey().toString() + ":");
            
            Set<Peer> peersWithResource = resource.getValue().getPeersWithResource();
            for (Peer peer : peersWithResource)
            {
                System.out.println(peer);
            }
            
            System.out.println(separator);
        }
    }
}
