/*
 * Peer.java
 *
 */

package notorrent.tracker;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import notorrent.util.UncheckedException;
import org.jdom.Element;

/**
 * Contains info about a peer.
 *
 * @author Howie
 */
public class Peer
{
    private static final String PEER_TAG    = "peer";
    private static final String ADDRESS_TAG = "address";
    private static final String PORT_TAG    = "port";
    
    private InetAddress address;
    
    public InetAddress getAddress()
    {
        return address;
    }
        
    public void setAddress(InetAddress address)
    {
        this.address = address;
    }
        
    private int port;
    
    public int getPort()
    {
        return port;
    }
    
    public void setPort(int port)
    {
        this.port = port;
    }
    
    /**
     * last time (in seconds since the unix epoch) any type of
     * message (keep alive, request, etc) was received for this peer
     */
    private int lastPingTime;
    
    /** Creates a new instance of Peer */
    public Peer(InetAddress address, int port, int lastPingTime)
    {
	this.address      = address;
	this.port         = port;
	this.lastPingTime = lastPingTime;
    }
    
    public Peer(InetAddress address, int port)
    {
	this(address, port, -1);
    }
    
    public Peer()
    {
        this.address    = null;
        this.port       = -1;
    }
    
    /**
     * returns the given peersList encoded as a JDOM Element with the given tagName.
     * this will look like:
     * <pre>&lt;tagName&gt;
     *	&lt;peer&gt;
     *	    &lt;address&gt;1.2.3.4&lt;/address&gt;
     *	    &lt;port&gt;5678&lt;/port&gt;
     *  &lt;/peer&gt;
     *	&lt;peer&gt;
     *	    &lt;address&gt;2.2.3.4&lt;/address&gt;
     *	    &lt;port&gt;3678&lt;/port&gt;
     *  &lt;/peer&gt;
     * &lt;/tagName&gt;</pre>
     */
    public static Element peersSetAsElement(String tagName, Set<Peer> peersSet)
    {
	Element peersListElement = new Element(tagName);
	
	for (Peer peer : peersSet)
	{
	    peersListElement.addContent(peer.toXMLElement());
	}
	
	return peersListElement;
    }
    
    /**
     * takes an Element whose subnodes are of the following form:
     *	&lt;peer&gt;
     *	    &lt;address&gt;1.2.3.4&lt;/address&gt;
     *	    &lt;port&gt;5678&lt;/port&gt;
     *  &lt;/peer&gt;
     * and sticks them in (and returns) a Set<Peer>.
     */
    public static Set<Peer> elementAsPeersSet(Element peersSetAsXML)
    {
	Set<Peer> peerList = new HashSet<Peer>();
	
	Iterator peerIter = (peersSetAsXML.getChildren(PEER_TAG)).iterator();
	Element peerElement;
	Peer peer;
	InetAddress peerAddress;
	int peerPort;
	
	while(peerIter.hasNext())
	{
	    peerElement = (Element)peerIter.next();
	    
	    Element addressElement  = peerElement.getChild(ADDRESS_TAG);
	    Element portElement	    = peerElement.getChild(PORT_TAG);
	    
	    try
	    {
		peerAddress = InetAddress.getByName(addressElement.getTextNormalize());
	    }
	    catch (UnknownHostException e)
	    {
		throw new UncheckedException(e);
	    }
	    peerPort = Integer.parseInt(portElement.getTextNormalize());
	    peer = new Peer(peerAddress, peerPort);
	    peerList.add(peer);
	}
	
	return peerList;
    }
    
    /**
     * returns this Peer encoded as a JDOM Element.
     * this will look like:
     * <pre>&lt;peer&gt;
     *	    &lt;address&gt;1.2.3.4&lt;/address&gt;
     *	    &lt;port&gt;5678&lt;/port&gt;
     *  &lt;/peer&gt;</pre>
     */
    private Element toXMLElement()
    {
	Element peerElement = new Element(PEER_TAG);
	
	Element addressElement = new Element(ADDRESS_TAG);
	addressElement.setText(address.getHostAddress());
	peerElement.addContent(addressElement);
	
	Element portElement = new Element(PORT_TAG);
	portElement.setText(Integer.toString(port));
	peerElement.addContent(portElement);
	
	return peerElement;
    }
    
    public String toString()
    {
	return address.getHostName() + "/" + address.getHostAddress() + ":" + port;
    }
    
    public boolean equals(Object obj)
    {
	if (this == obj)
	{
	    return true;
	}
	
	if (!(obj instanceof Peer))
	{
	    return false;
	}
	
	// object must be a Peer at this point
	Peer that = (Peer)obj;
	return this.address.equals(that.address) &&
		this.port == that.port;
    }
    
    public int hashCode()
    {
	int hash = 7;
	hash = 31 * hash + address.hashCode();
	hash = 31 * hash + port;
	return hash;
    }
    
    public static String peersListToString(List<Peer> peersList)
    {
	// should use StringWriter here instead
	String str = "";
	
	for (Peer peer : peersList)
	{
	    str += "\n\t\t<peer>" + peer + "\n\t\t</peer>";
	}
	
	return str;
    }
}
