/*
 * CommandLineArgs.java
 *
 */

package notorrent.tracker;

/**
 *
 * @author Howie
 */
public class CommandLineArgs
{
    /** Creates a new instance of CommandLineArgs */
    public CommandLineArgs()
    {
	// set default values
	this.timeout			    = 5 * 60;
	this.purgeFrequency		    = 10 * 60;
	this.stateFile			    = "statefile";  // file named "statefile" in local directory
	this.stateFileSaveFrequency	    = 5 * 60;
	this.numThreads			    = 2;
	this.portToListenOn		    = 4242;
	this.numTimeWindows		    = 3;
	this.timeWindowDurationInMinutes    = 60;
	this.numResourcesToTrack	    = 50;
    }
    
    private int numTimeWindows;
    
    /**
     * the number of time windows to keep track of.
     * i.e. if a time window is an hour long and numTimeWindows is 3,
     * we'll keep track of the number of requests in each of the last 3 hours.
     */
    public int getNumTimeWindows()
    {
	return numTimeWindows;
    }
    
    private int timeWindowDurationInMinutes;
    private int numResourcesToTrack;
    
    /**
     * # seconds with no contact with a client before declaring it "timed out,"
     * which means it won't be sent out in lists of which peers have a file,
     * and which means it will be purged from all lists the next time that's done
     */
    private int timeout;
    
    /**
     * how often (in seconds) to do a purge of all timed out peers
     */
    private int purgeFrequency;
    
    /**
     * where (path and filename) to save statefile
     */
    private String stateFile;
    
    /**
     * how often (in seconds) to save state data to file (specified by
     * stateFile parameter)
     */
    private int stateFileSaveFrequency;
    
    /**
     * number of HandlerThreads to spawn
     */
    private int numThreads;
    
    /**
     * port number on which to listen for client connections
     */
    private int portToListenOn;
    
    /**
     * sets tracker settings based on command line arguments.
     *
     * based on:
     * http://scv.bu.edu/Doc/Java/tutorial/java/cmdLineArgs/example/ParseCmdLine.java
     */
    public void parseCommandLineArgs(String[] args) throws NumberFormatException
    {
	int i = 0, j;
	String arg;
	boolean verboseModeIsOn = false;
	
	while (i < args.length && args[i].startsWith("-"))
	{
	    arg = args[i++];
	    
	    if (arg.equalsIgnoreCase("-verbose") || arg.equalsIgnoreCase("-v"))
	    {
		verboseModeIsOn = true;
	    }
	    else if (arg.equalsIgnoreCase("-help") || arg.equalsIgnoreCase("-h"))
	    {
		printHelpMessage();
		System.exit(0);
	    }
	    else if (arg.equalsIgnoreCase("-timeout"))
	    {
		if (i < args.length)
		{
		    timeout = Integer.parseInt(args[i++]);
		}
		else
		{
		    System.err.println("-timeout requires an integer argument");
		    System.exit(1);
		}
	    }
	    else if (arg.equalsIgnoreCase("-purgeFrequency"))
	    {
		if (i < args.length)
		{
		    purgeFrequency = Integer.parseInt(args[i++]);
		}
		else
		{
		    System.err.println("-purgeFrequency requires an integer argument");
		    System.exit(1);
		}
	    }
	    else if (arg.equalsIgnoreCase("-stateFile"))
	    {
		if (i < args.length)
		{
		    stateFile = args[i++];
		}
		else
		{
		    System.err.println("-stateFile requires a string argument");
		    System.exit(1);
		}
	    }
	    else if (arg.equalsIgnoreCase("-stateFileSaveFrequency"))
	    {
		if (i < args.length)
		{
		    stateFileSaveFrequency = Integer.parseInt(args[i++]);
		}
		else
		{
		    System.err.println("-stateFileSaveFrequency requires an integer argument");
		    System.exit(1);
		}
	    }
	    else if (arg.equalsIgnoreCase("-numThreads"))
	    {
		if (i < args.length)
		{
		    numThreads = Integer.parseInt(args[i++]);
		}
		else
		{
		    System.err.println("-numThreads requires an integer argument");
		    System.exit(1);
		}
	    }
	    else if (arg.equalsIgnoreCase("-portToListenOn") || arg.equalsIgnoreCase("-port") || arg.equalsIgnoreCase("-p"))
	    {
		if (i < args.length)
		{
		    portToListenOn = Integer.parseInt(args[i++]);
		}
		else
		{
		    System.err.println("-" + arg + " requires an integer argument");
		    System.exit(1);
		}
	    }
	}
	
	if (verboseModeIsOn)
	{
	    System.out.println("timeout                = " + timeout);
	    System.out.println("purgeFrequency         = " + purgeFrequency);
	    System.out.println("stateFile              = " + stateFile);
	    System.out.println("stateFileSaveFrequency = " + stateFileSaveFrequency);
	    System.out.println("numThreads             = " + numThreads);
	    System.out.println("portToListenOn         = " + portToListenOn);
	}
    }
    
    /**
     * prints help message describing possible command line args.
     * called if -h or -help options set
     */
    private void printHelpMessage()
    {
	String classname = "notorrent/tracker/Tracker";
	
	System.out.println("Usage: java -cp jdom.jar:NoTorrent.jar " + classname + " [-options]");
	System.out.println("\nwhere options include:");
	
	System.out.println("\n-v");
	System.out.println("-verbose");
	System.out.println("enable verbose output (show settings on startup)");
	
	System.out.println("\n-h");
	System.out.println("-help");
	System.out.println("show this help message");
	
	System.out.println("\n-timeout <timeout>");
	System.out.println("# seconds with no contact with a client before declaring it \"timed out,\" +" +
		" which means it won't be sent out in lists of which peers have a file, " +
		" and which means it will be purged from all lists the next time that's done");
	
	System.out.println("\n-purgeFrequency <purgeFrequency>");
	System.out.println("how often (in seconds) to do a purge of all timed out peers");
	
	System.out.println("\n-stateFile <stateFile>");
	System.out.println("where (path and filename) to save statefile");
	
	System.out.println("\n-stateFileSaveFrequency <stateFileSaveFrequency>");
	System.out.println("how often (in seconds) to save state data to file (specified by" +
		" stateFile parameter)");
	
	System.out.println("\n-numThreads <numThreads>");
	System.out.println("number of HandlerThreads to spawn");
	
	System.out.println("\n-p <portToListenOn>");
	System.out.println("\n-port <portToListenOn>");
	System.out.println("\n-portToListenOn <portToListenOn>");
	System.out.println("port to listen for client connections on");
    }
    
    public int getPortToListenOn()
    {
	return portToListenOn;
    }
    
    public int getNumThreads()
    {
	return numThreads;
    }
}
