/*
 * XMLEncodedMessage.java
 *
 * Created on December 9, 2005, 12:02 AM
 *
 * To change this template, choose Tools | Options and locate the template under
 * the Source Creation and Management node. Right-click the template and choose
 * Open. You can then make changes to the template in the Source Editor.
 */

package notorrent.messages;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringWriter;
import notorrent.util.UncheckedException;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * used to encode NoTorrent messages in XML
 *
 * @author hrv2101
 */
public class XMLEncodedMessage
{
    private Document messageDocument;
    private Element messageTypeElement;
    private Element messageContentsElement;
    
    // constants
    private static final String MESSAGE_TAG		= "message";
    private static final String MESSAGE_TYPE_TAG	= "messageType";
    private static final String MESSAGE_CONTENTS_TAG	= "messageContents";
    
    /** Creates a new instance of XMLEncodedMessage */
    public XMLEncodedMessage(String messageType)
    {
        Element rootElement = new Element(MESSAGE_TAG);
	messageDocument = new Document(rootElement);
	
	messageTypeElement = new Element(MESSAGE_TYPE_TAG);
	messageTypeElement.setText(messageType);
	rootElement.addContent(messageTypeElement);
	
	messageContentsElement = new Element(MESSAGE_CONTENTS_TAG);
	rootElement.addContent(messageContentsElement);
    }
    
    /**
     * Creates a new instance of XMLEncodedMessage based on the given
     * Document doc.  Wrapping a JDOM document in an XMLEncodedMessage
     * makes it easy to access the messageType and the messageContents.
     */
    public XMLEncodedMessage(Document doc)
    {
	messageDocument = doc;
	Element rootElement = messageDocument.getRootElement();
	
	messageTypeElement = rootElement.getChild(MESSAGE_TYPE_TAG);
	
	messageContentsElement = rootElement.getChild(MESSAGE_CONTENTS_TAG);
	
	if (messageTypeElement == null || messageContentsElement == null)
	{
	    throw new MalformedMessageException();
	}
    }
    
    /**
     * adds a node with the given fieldName and fieldText to this message's
     * messageContents node.
     */
    public void addField(String fieldName, String fieldText)
    {
	Element newFieldElement = new Element(fieldName);
	newFieldElement.setText(fieldText);
	messageContentsElement.addContent(newFieldElement);
    }
    
    /**
     * adds the given complexField to this message's messageContents node.
     * use this when the field to be added is something other than a String.
     * (use addField(String fieldName, String fieldText) for that.)
     * for example, if complexField represents
     * <pre>&lt;peerList&gt;
     *	    &lt;peer&gt;1.2.3.4&lt;/peer&gt;
     *	    &lt;peer&gt;1.2.3.5&lt;/peer&gt;
     * &lt;/peerList&gt;</pre>,
     * then that (peerList) node will be added to the messageContents node.
     */
    public void addField(Element complexField)
    {
	messageContentsElement.addContent(complexField);
    }
    
    /**
     * returns this XML-encoded message as a byte array.
     */
    public byte[] toByteArray()
    {
	XMLOutputter outputter = new XMLOutputter(Format.getCompactFormat());
	ByteArrayOutputStream baos = new ByteArrayOutputStream();
	
	try
	{
	    outputter.output(messageDocument, baos);
	}
	catch (IOException e)
	{
	    throw new UncheckedException(e);
	}
	
	return baos.toByteArray();
    }
    
    private String messageType()
    {
	return messageTypeElement.getTextNormalize();
    }
    
    /**
     * returns the message as an Message (e.g. PeerListRequest, PeerListResponse, etc)
     **/
    public Message toMessage()
    {
	String messageType = messageType();
	
	if (messageType.equals(MessageType.PEER_LIST_REQUEST))
	{
	    return new MessagePeerListRequest(messageContentsElement);
	}
	else if (messageType.equals(MessageType.PEER_LIST_RESPONSE))
	{
	    return new MessagePeerListResponse(messageContentsElement);
	}
	else if (messageType.equals(MessageType.I_HAVE_RESOURCE))
	{
	    return new MessageIHaveResource(messageContentsElement);
	}
	else if (messageType.equals(MessageType.ERROR))
	{
	    return new MessageError(messageContentsElement);
	}
	else if (messageType.equals(MessageType.RESOURCE_REQUEST))
	{
	    return new MessageResourceRequest(messageContentsElement);
	}
	else if (messageType.equals(MessageType.PEER_FAILED_TO_SERVE))
	{
	    return new MessagePeerFailedToServe(messageContentsElement);
	}        
	else
	{
	    throw new MalformedMessageException("Invalid message type");
	}	
    }
    
    public String toString()
    {
	XMLOutputter outputter = new XMLOutputter(Format.getPrettyFormat());
	StringWriter sw = new StringWriter();
	
	try
	{
	    outputter.output(messageDocument, sw);
	}
	catch (IOException e)
	{
	    throw new UncheckedException(e);
	}
	
	return sw.toString();
    }
}
