/*
 * MessagePeerListRequest.java
 *
 */

package notorrent.messages;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Set;
import notorrent.tracker.Peer;
import notorrent.tracker.StateInfo;
import notorrent.util.UncheckedException;
import org.jdom.Element;

/**
 * @author Howie
 */
public class MessagePeerListRequest extends Message
{
    private static final String REQUESTED_RESOURCE_TAG	    = "requestedResource";
    
    /** whether this message has been populated with values */
    private boolean messagePopulated	= false;
    
    private URI requestedResource	= null;
    
    public URI getRequestedResource()
    {
	return requestedResource;
    }
    
    /**
     * the messageContentsElement is an XML node containing this message's
     * fields (encoded in XML of course).
     */
    public MessagePeerListRequest(Element messageContentsElement)
    {
	decode(messageContentsElement);
    }
    
    public MessagePeerListRequest(URI requestedResource)
    {
	this.requestedResource	    = requestedResource;
	
	this.messagePopulated	    = true;
    }
    
    public MessagePeerListRequest(String requestedResource)
    {
	try
	{
	    this.requestedResource	= new URI(requestedResource);
	    
	    this.messagePopulated	= true;
	}
	catch (java.net.URISyntaxException e)
	{
	    throw new UncheckedException(e);
	}
    }
    
    /**
     * if a message is not received from client within timeout seconds,
     * client will be dropped from peer list.
     * encapsulate this in a method so that we can easily change how we determine it.
     */
    private int timeout()
    {
	return 60 * 10;
    }
    
    public Message handle()
    {
	if (!messagePopulated)
	    throw new MessageNotPopulatedException();
	
	Set<Peer> peersWithResource = StateInfo.getPeersWithResource(requestedResource);
	
	MessagePeerListResponse response = new MessagePeerListResponse(timeout(), peersWithResource, requestedResource);
	
	return response;
    }
       
    XMLEncodedMessage encodeAsXML()
    {
	XMLEncodedMessage encodedMessage = new XMLEncodedMessage(MessageType.PEER_LIST_REQUEST);
	
	encodedMessage.addField(REQUESTED_RESOURCE_TAG, requestedResource.toString());
	
	return encodedMessage;
    }
    
    /**
     * set this message's fields based on the subnodes of the given
     * messageContentsElement.
     */
    private void decode(Element messageContentsElement)
    {
	try
	{
	    String requestedResourceString 	= messageContentsElement.getChildTextTrim(REQUESTED_RESOURCE_TAG);
	    
	    if (requestedResourceString == null)
	    {
		throw new MalformedMessageException("a required field tag is missing");
	    }
	    
	    requestedResource	    = new URI(requestedResourceString);
	    
	    this.messagePopulated   = true;
	}
	catch (URISyntaxException e)
	{
	    throw new UncheckedException(e);
	}
    }
}
