/*
 * MessagePeerFailedToServe.java
 */

package notorrent.messages;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import notorrent.tracker.Peer;
import notorrent.tracker.StateInfo;
import notorrent.util.UncheckedException;
import org.jdom.Element;

/**
 *
 * @author hrv2101
 */
public class MessagePeerFailedToServe extends Message
{
    private static final String RESOURCE_TAG        = "resourceUri";
    private static final String PEER_ADDRESS_TAG    = "peerAddress";
    private static final String PEER_PORT_TAG       = "peerPort";
    
    /** whether this message has been populated with values */
    private boolean messagePopulated	= false;
    
    private URI resource	= null;
    
    public URI getResource()
    {
        return resource;
    }
    
    /** the peer's address */
    private InetAddress peerAddress	= null;
    
    public InetAddress getPeerAddress()
    {
        return peerAddress;
    }
    
    private int peerPort	= -1;
    
    public int getPeerPort()
    {
        return peerPort;
    }
    
    /**
     * the messageContentsElement is an XML node containing this message's
     * fields (encoded in XML of course).
     */
    public MessagePeerFailedToServe(Element messageContentsElement)
    {
        decode(messageContentsElement);
    }
    
    public MessagePeerFailedToServe(
            URI resource,
            InetAddress peerAddress,
            int peerPort)
    {
        this.resource	    = resource;
        this.peerAddress		    = peerAddress;
        this.peerPort   = peerPort;
        
        this.messagePopulated	    = true;
    }
    
    public MessagePeerFailedToServe(
            String resource,
            InetAddress peerAddress,
            int peerPort)
    {
        try
        {
            this.resource	= new URI(resource);
            this.peerAddress		= peerAddress;
            this.peerPort  = peerPort;
            
            this.messagePopulated	= true;
        }
        catch (java.net.URISyntaxException e)
        {
            throw new UncheckedException(e);
        }
    }
    
    public Message handle()
    {
        if (!messagePopulated)
            throw new MessageNotPopulatedException();
        
        StateInfo.peerFailedToServe(resource, new Peer(peerAddress, peerPort));
        
        System.out.println("in MessagePeerFailedToServe.handle().  noted that peer failed to serve.  no response message will be sent.");
        
        return null;
    }
    
    XMLEncodedMessage encodeAsXML()
    {
        XMLEncodedMessage encodedMessage = new XMLEncodedMessage(MessageType.PEER_FAILED_TO_SERVE);
        
        encodedMessage.addField(RESOURCE_TAG, resource.toString());
        encodedMessage.addField(PEER_ADDRESS_TAG, peerAddress.getHostAddress());
        encodedMessage.addField(PEER_PORT_TAG, Integer.toString(peerPort));
        
        return encodedMessage;
    }
    
    /**
     * set this message's fields based on the subnodes of the given
     * messageContentsElement.
     */
    private void decode(Element messageContentsElement)
    {
        try
        {
            String resourceString 	= messageContentsElement.getChildTextTrim(RESOURCE_TAG);
            String peerAddressString		= messageContentsElement.getChildTextTrim(PEER_ADDRESS_TAG);
            String peerPortString	= messageContentsElement.getChildTextTrim(PEER_PORT_TAG);
            
            if (resourceString == null ||
                    peerAddressString == null ||
                    peerPortString == null)
            {
                throw new MalformedMessageException("a required field tag is missing");
            }
            
            resource	= new URI(resourceString);
            peerAddress = InetAddress.getByName(peerAddressString);
            peerPort    = Integer.parseInt(peerPortString);
            
            this.messagePopulated   = true;
        }
        catch (URISyntaxException e)
        {
            throw new UncheckedException(e);
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
    }
}
