/*
 * MessageIHaveResource.java
 *
 */

package notorrent.messages;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import notorrent.tracker.Peer;
import notorrent.tracker.StateInfo;
import notorrent.util.UncheckedException;
import org.jdom.Element;

/**
 *
 * @author Howie
 */
public class MessageIHaveResource extends Message
{
    private static final String RESOURCE_TAG		     = "resourceUri";
    private static final String MY_ADDRESS_TAG		     = "myAddress";
    private static final String PORT_TO_LISTEN_FOR_PEERS_TAG = "portToListenForPeers";
    
    /** whether this message has been populated with values */
    private boolean messagePopulated	= false;
    
    private URI resource	= null;
    
    public URI getResource()
    {
	return resource;
    }
    
    /** the client's address */
    private InetAddress myAddress	= null;
    
    public InetAddress getMyAddress()
    {
	return myAddress;
    }
    
    private int portToListenForPeers	= -1;
    
    public int getPortToListenForPeers()
    {
	return portToListenForPeers;
    }
    
    /**
     * the messageContentsElement is an XML node containing this message's
     * fields (encoded in XML of course).
     */
    public MessageIHaveResource(Element messageContentsElement)
    {
	decode(messageContentsElement);
    }
    
    public MessageIHaveResource(
	    URI resource,
	    InetAddress myAddress,
	    int portToListenForPeers)
    {
	this.resource	    = resource;
	this.myAddress		    = myAddress;
	this.portToListenForPeers   = portToListenForPeers;
	
	this.messagePopulated	    = true;
    }
    
    public MessageIHaveResource(
	    String resource,
	    InetAddress myAddress,
	    int portToListenForPeers)
    {
	try
	{
	    this.resource	= new URI(resource);
	    this.myAddress		= myAddress;
	    this.portToListenForPeers   = portToListenForPeers;
	    
	    this.messagePopulated	= true;
	}
	catch (java.net.URISyntaxException e)
	{
	    throw new UncheckedException(e);
	}
    }
    
    public Message handle()
    {
	if (!messagePopulated)
	    throw new MessageNotPopulatedException();
	
	StateInfo.iHaveResource(resource, new Peer(myAddress, portToListenForPeers));
	
	System.out.println("in MessageIHaveResource.handle().  noted that client has resource.  no response message will be sent.");
	
	return null;
    }
    
    XMLEncodedMessage encodeAsXML()
    {
	XMLEncodedMessage encodedMessage = new XMLEncodedMessage(MessageType.I_HAVE_RESOURCE);
	
	encodedMessage.addField(RESOURCE_TAG, resource.toString());
	encodedMessage.addField(MY_ADDRESS_TAG, myAddress.getHostAddress());
	encodedMessage.addField(PORT_TO_LISTEN_FOR_PEERS_TAG, Integer.toString(portToListenForPeers));
	
	return encodedMessage;
    }
    
    /**
     * set this message's fields based on the subnodes of the given
     * messageContentsElement.
     */
    private void decode(Element messageContentsElement)
    {
	try
	{
	    String resourceString 	= messageContentsElement.getChildTextTrim(RESOURCE_TAG);
	    String myAddressString		= messageContentsElement.getChildTextTrim(MY_ADDRESS_TAG);
	    String portToListenForPeersString	= messageContentsElement.getChildTextTrim(PORT_TO_LISTEN_FOR_PEERS_TAG);
	    
	    if (resourceString == null ||
		    myAddressString == null ||
		    portToListenForPeersString == null)
	    {
		throw new MalformedMessageException("a required field tag is missing");
	    }
	    
	    resource	    = new URI(resourceString);
	    myAddress		    = InetAddress.getByName(myAddressString);
	    portToListenForPeers    = Integer.parseInt(portToListenForPeersString);
	    
	    this.messagePopulated   = true;
	}
	catch (URISyntaxException e)
	{
	    throw new UncheckedException(e);
	}
	catch (IOException e)
	{
	    throw new UncheckedException(e);
	}
    }
}
