/*
 * Message.java
 *
 */

package notorrent.messages;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketException;
import notorrent.util.UncheckedException;
import org.jdom.Document;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;

/**
 * Message superclass.  holds methods common to all messages.
 *
 * @author Howie
 */
public abstract class Message
{
    /** maximum length of a message in bytes */
//    private final static int MAX_MESSAGE_SIZE = 1024;
    
    public static Message decode(byte[] messageAsBytes)
    {
        return Message.decode(new ByteArrayInputStream(messageAsBytes));
    }
    
    /**
     * converts the inputStream into a Message.
     * throws an exception if there's no data to read or if the
     * inputStream is not a valid Message.
     */
    public static Message decode(InputStream inputStream)
    {
	try
	{
	    // build a JDOM Document from the XML in the input byte stream
	    SAXBuilder builder = new SAXBuilder();
	    Document doc = builder.build(inputStream);
	    
	    // create an XMLEncodedMessage based on the input Document
	    XMLEncodedMessage encodedMessage = new XMLEncodedMessage(doc);
	    
	    // returns the message as an Message (e.g. PeerListRequest, PeerListResponse, etc)
	    return encodedMessage.toMessage();
	}
	catch (JDOMException e)
	{
	    throw new UncheckedException(e);
	}
	catch (IOException e)
	{
	    throw new UncheckedException(e);
	}
    }
    
    /**
     * handles this request message.
     * if there is a response message to be sent, it is returned.
     * if there is no response to be sent, null is returned.
     */
    public abstract Message handle();
    
    abstract XMLEncodedMessage encodeAsXML();

    /**
     * returns an XML-encoded version of this message as a byte[]
     */
    public byte[] encode()
    {
	XMLEncodedMessage encodedMessage = encodeAsXML();
	
	return encodedMessage.toByteArray();
    }
    
    public String toString()
    {
	XMLEncodedMessage encodedMessage = encodeAsXML();
	
	return encodedMessage.toString();
    }
}
