/*
 * CommandLineArgs.java
 *
 */

package notorrent.client;

/**
 *
 * @author Howie
 */
public class CommandLineArgs
{
    /** Creates a new instance of CommandLineArgs */
    public CommandLineArgs()
    {
        // set default values
        this.numClientServerThreads         = 2;
        this.numClientProxyThreads          = 2;
        this.clientProxyPort                = 6901;
        this.firstClientServerPortToTry     = 4001;
        this.maxNumClientServerPortsToTry   = 10;
        this.trackerAddress                 = "localhost";
        this.trackerPort                    = 4242;
    }
    
    public int getNumClientServerThreads()
    {
        return numClientServerThreads;
    }
    private int numClientServerThreads;
    
    public int getNumClientProxyThreads()
    {
        return numClientProxyThreads;
    }
    private int numClientProxyThreads;
    
    public int getClientProxyPort()
    {
        return clientProxyPort;
    }
    private int clientProxyPort;
    
    /** the first port the client server should try to listen on.  if it fails, keep incrementint the port until an open port is found. */
    public int getFirstClientServerPortToTry()
    {
        return firstClientServerPortToTry;
    }
    private int firstClientServerPortToTry;
    
    /**
     * we start looking for open ports at port firstClientServerPortToTry.  
     * we look for an open port on at most the next maxNumPortsToTry - 1 ports. 
     */
    public int getMaxNumClientServerPortsToTry()
    {
        return maxNumClientServerPortsToTry;
    }
    private int maxNumClientServerPortsToTry;
    
    public String getTrackerAddress()
    {
        return trackerAddress;
    }
    private String trackerAddress;
    
    public int getTrackerPort()
    {
        return trackerPort;
    }
    private int trackerPort;
    
    /**
     * sets tracker settings based on command line arguments.
     *
     * based on:
     * http://scv.bu.edu/Doc/Java/tutorial/java/cmdLineArgs/example/ParseCmdLine.java
     */
    public void parseCommandLineArgs(String[] args) throws NumberFormatException
    {
        int i = 0, j;
        String arg;
        boolean verboseModeIsOn = false;
        
        while (i < args.length && args[i].startsWith("-"))
        {
            arg = args[i++];
            
            if (arg.equalsIgnoreCase("-verbose") || arg.equalsIgnoreCase("-v"))
            {
                verboseModeIsOn = true;
            }
            else if (arg.equalsIgnoreCase("-help") || arg.equalsIgnoreCase("-h"))
            {
                printHelpMessage();
                System.exit(0);
            }
            else if (arg.equalsIgnoreCase("-numClientServerThreads") || arg.equalsIgnoreCase("-cst"))
            {
                if (i < args.length)
                {
                    numClientServerThreads = Integer.parseInt(args[i++]);
                }
                else
                {
                    System.err.println("-numClientServerThreads requires an integer argument");
                    System.exit(1);
                }
            }
            else if (arg.equalsIgnoreCase("-numClientProxyThreads") || arg.equalsIgnoreCase("-cpt"))
            {
                if (i < args.length)
                {
                    numClientProxyThreads = Integer.parseInt(args[i++]);
                }
                else
                {
                    System.err.println("-numClientProxyThreads requires an integer argument");
                    System.exit(1);
                }
            }
            else if (arg.equalsIgnoreCase("-clientProxyPort") || arg.equalsIgnoreCase("-cp"))
            {
                if (i < args.length)
                {
                    clientProxyPort = Integer.parseInt(args[i++]);
                }
                else
                {
                    System.err.println("-[clientProxyPort|cp] requires an integer argument");
                    System.exit(1);
                }
            }
            else if (arg.equalsIgnoreCase("-firstClientServerPortToTry") || arg.equalsIgnoreCase("-cs"))
            {
                if (i < args.length)
                {
                    firstClientServerPortToTry = Integer.parseInt(args[i++]);
                }
                else
                {
                    System.err.println("-[firstClientServerPortToTry|cs] requires an integer argument");
                    System.exit(1);
                }
            }
            else if (arg.equalsIgnoreCase("-maxNumClientServerPortsToTry") || arg.equalsIgnoreCase("-csp"))
            {
                if (i < args.length)
                {
                    maxNumClientServerPortsToTry = Integer.parseInt(args[i++]);
                }
                else
                {
                    System.err.println("-[maxNumClientServerPortsToTry|csp] requires an integer argument");
                    System.exit(1);
                }
            }
            else if (arg.equalsIgnoreCase("-trackerAddress") || arg.equalsIgnoreCase("-ta"))
            {
                if (i < args.length)
                {
                    trackerAddress = args[i++];
                }
                else
                {
                    System.err.println("-[trackerAddress|ta] requires an argument");
                    System.exit(1);
                }
            }
            else if (arg.equalsIgnoreCase("-trackerPort") || arg.equalsIgnoreCase("-tp"))
            {
                if (i < args.length)
                {
                    trackerPort = Integer.parseInt(args[i++]);
                }
                else
                {
                    System.err.println("-[trackerPort|tp] requires an integer argument");
                    System.exit(1);
                }
            }
        }
        
        if (verboseModeIsOn)
        {
            System.out.println("numClientServerThreads     = " + numClientServerThreads);
            System.out.println("numClientProxyThreads      = " + numClientProxyThreads);
            System.out.println("clientProxyPort            = " + clientProxyPort);
            System.out.println("firstClientServerPortToTry = " + firstClientServerPortToTry);
            System.out.println("maxNumClientServerPortsToTry = " + maxNumClientServerPortsToTry);
            System.out.println("trackerAddress             = " + trackerAddress);
            System.out.println("trackerPort                = " + trackerPort);
        }
    }
    
    /**
     * prints help message describing possible command line args.
     * called if -h or -help options set
     */
    private void printHelpMessage()
    {
        String classname = "notorrent/client/Client";
        
        System.out.println("Usage: java -cp jdom.jar:NoTorrent.jar " + classname + " [-options]");
        System.out.println("\nwhere options include:");
        
        System.out.println("\n-v");
        System.out.println("-verbose");
        System.out.println("enable verbose output (show settings on startup)");
        
        System.out.println("\n-h");
        System.out.println("-help");
        System.out.println("show this help message");
        
        System.out.println("\n-[cst|numClientServerThreads] <numClientServerThreads>");
        System.out.println("\n-[cpt|numClientProxyThreads] <numClientProxyThreads>");
        System.out.println("\n-[cp|clientProxyPort] <clientProxyPort>");
        System.out.println("\n-[cs|firstClientServerPortToTry] <firstClientServerPortToTry>");
        System.out.println("\n-[csp|maxNumClientServerPortsToTry] <maxNumClientServerPortsToTry>");
        System.out.println("\n-[t|trackerAddress] <trackerAddress>");
        System.out.println("\n-[tp|trackerPort] <trackerPort>");
    }
}
