/*
 * ClientServerThread.java
 *
 */

package notorrent.client;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URI;
import notorrent.messages.Message;
import notorrent.messages.MessageResourceRequest;
import notorrent.util.Debug;
import notorrent.util.UncheckedException;

/**
 * Waits for a connection from a peer; handles that peer connection; and
 * then waits for another connection.
 *
 * @author hrv2101
 */
public class ClientServerThread implements Runnable
{
    /** to keep track of the separate threads */
    private static int nextThreadId	= 0;
    private int threadId		= -1;
    
    private Client parentClient		= null;
    private ServerSocket serverSocket	= null;
    
    /**
     * Creates a new instance of ClientServerThread
     */
    public ClientServerThread(Client parentClient)
    {
        this.threadId = nextThreadId++;
        this.parentClient = parentClient;
        this.serverSocket = parentClient.getClientServerSocket();
    }
    
    public void run()
    {
        Socket peerSocket;
        
        while (true)
        {
            try
            {
                printStatus("waiting for request from a peer...");
                peerSocket = serverSocket.accept();
                
                printStatus("request from a peer received.  going to handle it now.");
                handleRequest(peerSocket);
                
                peerSocket.close();
                printStatus("request by peer handled");
            }
            catch (IOException e)
            {
                throw new UncheckedException(e);
            }
        }
    }
    
    private void handleRequest(Socket peerSocket)
    {
        try
        {
            // read peer's MessageResourceRequest
            URI requestedResourceURI = getURIOfPeersResourceRequest(peerSocket);

            if (requestedResourceURI == null)
            {
                // if invalid URI, we do nothing and just close the socket (in run() after handleRequest() has terminated)
                return;
            }

            // check if requested resource is in my cache
            byte[] requestedResource = ClientCache.requestResourceFromCache(requestedResourceURI);

            if (requestedResource == null)
            {
                // if resource is in my cache, we do nothing and just close the socket (in run() after handleRequest() has terminated)
                return;
            }

            // since the resource is in my cache, send it to peer
            sendResourceToPeer(peerSocket, requestedResource, requestedResourceURI);
        }
        catch (UncheckedException e)
        {
            Debug.printDebug(e);
        }
    }
    
    private URI getURIOfPeersResourceRequest(Socket peerSocket)
    {
        URI requestedResource = null;
        
        try
        {
            InputStream inputStream = peerSocket.getInputStream();
            
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            
            int BUFSIZE = 32;
            int recvMsgSize;
            byte[] byteBuffer = new byte[BUFSIZE];
            
            while ((recvMsgSize = inputStream.read(byteBuffer)) != -1)
            {
                baos.write(byteBuffer, 0,  recvMsgSize);
            }
            
            MessageResourceRequest peersResourceRequestMessage = (MessageResourceRequest)Message.decode(baos.toByteArray());
            System.out.println("received message from peer: " + peersResourceRequestMessage);
            
            requestedResource = peersResourceRequestMessage.getRequestedResource();
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        
        return requestedResource;
    }
    
    private void sendResourceToPeer(Socket peerSocket, byte[] requestedResource, URI requestedResourceURI)
    {
        try
        {
            printStatus("sending resource '" + requestedResourceURI.toString() + "' to peer.");
            BufferedOutputStream outputStream = new BufferedOutputStream(peerSocket.getOutputStream());
            
            outputStream.write(requestedResource);
            outputStream.flush();
            printStatus("sent resource '" + requestedResourceURI.toString() + "' to peer.");
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
    }
    
    private void printStatus(String message)
    {
        System.out.println("ClientServerThread #" + threadId + ": " + message);
    }
}
