/*
 * ClientProxyThread.java
 */

package notorrent.client;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import notorrent.messages.Message;
import notorrent.messages.MessageIHaveResource;
import notorrent.messages.MessagePeerFailedToServe;
import notorrent.messages.MessagePeerListRequest;
import notorrent.messages.MessagePeerListResponse;
import notorrent.messages.MessageResourceRequest;
import notorrent.tracker.Peer;
import notorrent.util.Constants;
import notorrent.util.Debug;
import notorrent.util.MethodNotImplementedException;
import notorrent.util.UncheckedException;

/**
 *
 * @author hrv2101
 */
public class ClientProxyThread implements Runnable
{
    /** to keep track of the separate threads */
    private static int nextThreadId	= 0;
    private int threadId		= -1;
    
    private Client parentClient		= null;
    private ServerSocket serverSocket	= null;
    
    /**
     * Creates a new instance of ClientProxyThread
     */
    public ClientProxyThread(Client parentClient)
    {
        this.threadId = nextThreadId++;
        this.parentClient = parentClient;
        this.serverSocket = parentClient.getClientProxySocket();
    }
    
    public void run()
    {
        Socket browserSocket;
        
        while (true)
        {
            try
            {
                printStatus("waiting for request from a browser...");
                browserSocket = serverSocket.accept();
                
                printStatus("request from a browser received.  going to handle it now.");
                handleRequest(browserSocket);
                
                browserSocket.close();
                printStatus("request by browser handled");
            }
            catch (IOException e)
            {
                throw new UncheckedException(e);
            }
        }
    }
    
    private void handleRequest(Socket browserSocket)
    {
        byte[] retrievedResource = null;   // the resource/file the user requested
        
        BrowsersResourceRequest usersRequest = null;
        
        FailureReasons reasonsForFailure = new FailureReasons();  // set this if there is an error.  use it in the error page.
        
        try
        {
            usersRequest = BrowsersResourceRequest.readUsersRequest(browserSocket);
            
            URI requestedUri = usersRequest.getURI();  // throws URISyntaxException if URI string contains an illegal character such as '|'
            System.out.println("requestedUri is " + (requestedUri == null ? "null" : requestedUri));
            
            retrievedResource = requestResourceFromOriginServer(requestedUri, reasonsForFailure);
            
            // retrievedResource will be null if it was not successfully retrieved from the origin server
            if (retrievedResource != null)
            {
                cacheRetrievedResourceAndInformTracker(retrievedResource, requestedUri);
                
                // only modify the resource after caching it
                retrievedResource = prependStringToResource(retrievedResource, Constants.SERVED_VIA_NT_FROM_ORIGIN_SERVER);
                
                printStatus("resource served via origin server");
            }
            else
            {
                // look for the content in this client's cache
                retrievedResource = ClientCache.requestResourceFromCache(requestedUri);
                
                if (retrievedResource != null)
                {
                    retrievedResource = prependStringToResource(retrievedResource, Constants.SERVED_VIA_NT_FROM_CACHE);
                    printStatus("resource served via local cache");
                }
            }
            
            // retrievedResource will be null if it was not successfully retrieved from the origin server or the cache
            if (retrievedResource == null)
            {
                Peer peerIGotResourceFrom = new Peer(); // to be used in the prepend status message below
                
                // ask other peers for the resource
                retrievedResource = requestResourceFromPeers(requestedUri, reasonsForFailure, peerIGotResourceFrom);
                
                if (retrievedResource != null) // resource was successfully retrieved from a peer
                {
                    cacheRetrievedResourceAndInformTracker(retrievedResource, requestedUri);
                    
                    // only modify the resource after caching it
                    String prependString = Constants.SERVED_VIA_NT_FROM_PEER + (peerIGotResourceFrom == null ? "" : " (" + peerIGotResourceFrom.toString() + ")");
                    retrievedResource = prependStringToResource(retrievedResource, prependString);
                    
                    printStatus("resource served via a NoTorrent peer");
                }
            }
        }
        catch (URISyntaxException e)
        {
            reasonsForFailure.setReasonUnknownFailed(e.toString());
            printStatus("Error parsing the browserRequest for the requested URL.");
        }
        catch (Exception e)
        {
            reasonsForFailure.setReasonUnknownFailed(e.toString());
            Debug.printDebug(e);
        }
        
        // retrievedResource will be null if it was not successfully retrieved from the origin server, the cache, or a peer
        if (retrievedResource == null)
        {
            // since file not retrieved, send browser an error page
            retrievedResource = getResourceNotFoundErrorPage(usersRequest, reasonsForFailure);
//	    retrievedResource = getBytesFromFile(Constants.RESOURCE_COULD_NOT_BE_RETRIEVED_FILENAME);
        }
        
        sendResponseToBrowser(retrievedResource, browserSocket);
    }
    
    private void printStatus(String message)
    {
        System.out.println("ClientProxyThread #" + threadId + ": " + message);
    }
    
    /**
     * try to get the resource for the requestedURI from the origin server.
     * upon success, returns byte[] containing file's contents.
     * upon failure, returns null.
     */
    private byte[] requestResourceFromOriginServer(URI requestedURI, FailureReasons reasonsForFailure)
    {
        InputStream urlInputStream = null;
        
        try
        {
            urlInputStream = requestedURI.toURL().openStream();
//	    URL url = requestedURI.toURL();
//
//	    System.out.println("url is " + (url == null ? "null" : url));
//
////	    urlInputStream = url.openStream();
//	    URLConnection urlConn = url.openConnection();
//
//	    System.out.println("urlConn is " + (urlConn == null ? "null" : "not null"));
//
//	    try
//	    {
//	    urlInputStream = urlConn.getInputStream();
//	    }
//	    catch (Exception e)
//	    {
//		System.out.println("exception thrown while trying \"urlConn.getInputStream();\"");
//		e.printStackTrace();
//	    }
//
//	    System.out.println("urlInputStream is " + (urlInputStream == null ? "null" : "not null"));
            
            
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            
            int BUFSIZE = 32;
            int recvMsgSize;
            byte[] byteBuffer = new byte[BUFSIZE];
            
            while ((recvMsgSize = urlInputStream.read(byteBuffer)) != -1)
            {
                baos.write(byteBuffer, 0,  recvMsgSize);
            }
            
            byte[] retrievedResource = baos.toByteArray();
            
            if (retrievedResource.length == 0)
            {
                // no resource retrieved
                return null;
            }
            else
            {
                return retrievedResource;
            }
        }
//	catch (UnknownHostException e)
//	{
//	    printStatus("ERROR   : '" + requestedUrl + "' not found via origin server.");
//	}
        catch (FileNotFoundException e)
        {
            reasonsForFailure.setReasonOriginServerFailed(e.toString());
            // requestedURI not found on origin server
            return null;
        }
        catch (MalformedURLException e)
        {
            reasonsForFailure.setReasonOriginServerFailed(e.toString());
            Debug.printDebug(e);
            return null;
        }
        catch (IOException e)
        {
            reasonsForFailure.setReasonOriginServerFailed(e.toString());
            Debug.printDebug(e);
            return null;
        }
        catch (IllegalArgumentException e)
        {
            reasonsForFailure.setReasonOriginServerFailed(e.toString());
            Debug.printDebug(e);
            return null;
        }
        finally
        {
            try
            {
                if (urlInputStream != null)
                {
                    urlInputStream.close();
                }
            }
            catch (IOException e)
            {
                throw new UncheckedException(e);
            }
        }
    }
    
    private byte[] requestResourceFromPeers(URI requestedUri, FailureReasons reasonsForFailure, Peer peerIGotResourceFrom)
    {
        // ask tracker which peers have this resource
        List<Peer> peerList = askTrackerForPeersWithResource(requestedUri, reasonsForFailure);
        
        // keep asking peers for the resource until either I successfully receive
        // the resource from a peer, or all peers have been asked
        while (!peerList.isEmpty())
        {
            Peer peerToAsk = pickAndRemoveARandomPeer(peerList);
            
            byte[] requestedResource = requestResourceFromPeer(peerToAsk, requestedUri);
            
            if (requestedResource != null)
            {                
                peerIGotResourceFrom.setAddress(peerToAsk.getAddress());
                peerIGotResourceFrom.setPort(peerToAsk.getPort());
                return requestedResource;
            }
            else
            {
                // inform tracker that the peer failed to server the resource
                informTrackerThatPeerFailedToServe(requestedUri, peerToAsk);
            }
        }
        
        // no peers had the resource.
        if (reasonsForFailure.getReasonPeersFailed() == null)  // make sure tracker didn't already set the error reason
        {
            reasonsForFailure.setReasonPeersFailed("No peers had the requested resource.");
        }
        
        return null;
    }
    
    /**
     * returns either the requested resource as a byte[]
     * or null if the peer didn't have the resource or there was an exception.
     */
    private byte[] requestResourceFromPeer(Peer peerToAsk, URI requestedUri)
    {
        try
        {
            Message resourceRequestMessage = new MessageResourceRequest(requestedUri);
            //	System.out.println(".getMyAddress():" + ((MessagePeerListRequest)peerListRequestMessage).getMyAddress().toString());
            //	System.out.println("InetAddress.getLocalHost():" + InetAddress.getLocalHost().toString());
            
            String peerString = peerToAsk.getAddress().toString() + ":" + peerToAsk.getPort() ;
            
            System.out.println("Message to be sent to peer '" + peerString + "':\n" + resourceRequestMessage);
            
            // connect to peer
            Socket peerSocket = new Socket(peerToAsk.getAddress(), peerToAsk.getPort());
            System.out.println("Connected to peer (" + peerString + ")...sending PeerListRequest message");
            
            BufferedOutputStream outputStream = new BufferedOutputStream(peerSocket.getOutputStream());
            
            // send the encoded message to the server
            outputStream.write(resourceRequestMessage.encode());
            outputStream.flush();
            
            // closing of output stream indicates end of message
            //	outputStream.close();
            peerSocket.shutdownOutput();
            
            InputStream inputStream = peerSocket.getInputStream();
            
            // receive response message
            //	MessagePeerListResponse peerListResponseMessage = (MessagePeerListResponse)Message.decode(inputStream);
            //	socket.close();
            //	System.out.println("Message received:\n" + peerListResponseMessage);
            
            System.out.println("waiting for magic...");
            
            int BUFSIZE = 32;
            int recvMsgSize;
            byte[] byteBuffer = new byte[BUFSIZE];
            
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            
            while ((recvMsgSize = inputStream.read(byteBuffer)) != -1)
            {
                baos.write(byteBuffer, 0,  recvMsgSize);
            }
            
            System.out.println("size of byte array received from peer: " + baos.size());
            
            peerSocket.close();
            
            if (baos.size() == 0)
            {
                // the peer didn't have the resource, so it closed the connection.
                // we return null here to indicate that the peer didn't have the resource.
                return null;
            }
            else
            {
                // the peer did have the resource and sent it to us.
                // we return the received byte[]
                return baos.toByteArray();
            }
        }
        catch (ConnectException e)
        {
            System.err.println("Could not connect to peer '" + peerToAsk.getAddress().toString() + ":" + peerToAsk.getPort() + "'.  Peer is probably down.");
//            Debug.printDebug(e);
            return null;
        }
        catch (IOException e)
        {
            Debug.printDebug(e);
            return null;
        }
    }
    
    private Peer pickAndRemoveARandomPeer(List<Peer> peerList)
    {
        //  just remove the first element "for now"
        return peerList.remove(0);
    }
    
    /**
     * asks tracker which Peers claim to have the given resource.
     * returns a List<Peer> of those Peers.
     * if no Peers have resource or if there is some exception, an empty List<Peer>
     * will be returned.
     */
    private List<Peer> askTrackerForPeersWithResource(URI requestedUri, FailureReasons reasonsForFailure)
    {
        List<Peer> peersWithResource = new ArrayList<Peer>();
        
        try
        {
            Message peerListRequestMessage = new MessagePeerListRequest(requestedUri);
            //	System.out.println(".getMyAddress():" + ((MessagePeerListRequest)peerListRequestMessage).getMyAddress().toString());
            //	System.out.println("InetAddress.getLocalHost():" + InetAddress.getLocalHost().toString());
            
            System.out.println("Message to be sent to tracker:\n" + peerListRequestMessage);
            
            // connect to tracker
            Socket trackerSocket = new Socket(parentClient.getTrackerAddress(), parentClient.getTrackerPort());
            System.out.println("Connected to tracker...sending PeerListRequest message");
            
            BufferedOutputStream outputStream = new BufferedOutputStream(trackerSocket.getOutputStream());
            
            
            // send the encoded message to the server
            outputStream.write(peerListRequestMessage.encode());
            outputStream.flush();
            
            // closing of output stream indicates end of message
            //	outputStream.close();
            trackerSocket.shutdownOutput();
            
            InputStream inputStream = trackerSocket.getInputStream();
            
            
            // receive response message
            //	MessagePeerListResponse peerListResponseMessage = (MessagePeerListResponse)Message.decode(inputStream);
            //	socket.close();
            //	System.out.println("Message received:\n" + peerListResponseMessage);
            
            System.out.println("waiting for magic...");
            
            int BUFSIZE = 32;
            int recvMsgSize;
            byte[] byteBuffer = new byte[BUFSIZE];
            
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            
            while ((recvMsgSize = inputStream.read(byteBuffer)) != -1)
            {
                baos.write(byteBuffer, 0,  recvMsgSize);
            }
            
            MessagePeerListResponse peerListResponseMessage = (MessagePeerListResponse)Message.decode(baos.toByteArray());
            
            peersWithResource = peerListResponseMessage.getPeersWithResourceList();
            
            //	Message responseMessage = Message.decode(inputStream);
            trackerSocket.close();
            System.out.println("Message received from tracker:\n" + peerListResponseMessage + "\n");
            
            //	int timeout = peerListResponseMessage.getTimeout();
            //	URI requestedResource = peerListResponseMessage.getRequestedResource();
        }
        catch (IOException e)
        {
            reasonsForFailure.setReasonPeersFailed("Failed to get peer list from tracker.  Perhaps tracker is down. (details: " + e.toString() + ")");
            Debug.printDebug(e);
        }
        
        return peersWithResource;
    }
    
    private void sendResponseToBrowser(byte[] retrievedResource, Socket browserSocket)
    {
        try
        {
            BufferedOutputStream outputStream = new BufferedOutputStream(browserSocket.getOutputStream());
            
            outputStream.write(retrievedResource);
            outputStream.flush();
            
//	    System.out.println("content to return to browser:\n" + new String(retrievedResource));
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
    }
    
    /**
     * copied from:
     * http://javaalmanac.com/egs/java.io/File2ByteArray.html
     *
     * Returns the contents of the file in a byte array.
     */
    private static byte[] getBytesFromFile(String filename)
    {
        byte[] bytes = null;
        
        try
        {
            File file = new File(filename);
            
            InputStream is;
            
            is = new FileInputStream(file);
            
            // Get the size of the file
            long length = file.length();
            
            // You cannot create an array using a long type.
            // It needs to be an int type.
            // Before converting to an int type, check
            // to ensure that file is not larger than Integer.MAX_VALUE.
            if (length > Integer.MAX_VALUE)
            {
                // File is too large
            }
            
            // Create the byte array to hold the data
            bytes = new byte[(int)length];
            
            // Read in the bytes
            int offset = 0;
            int numRead = 0;
            while (offset < bytes.length
                    && (numRead=is.read(bytes, offset, bytes.length-offset)) >= 0)
            {
                offset += numRead;
            }
            
            // Ensure all the bytes have been read in
            if (offset < bytes.length)
            {
                throw new UncheckedException(new IOException("Could not completely read file "+file.getName()));
            }
            
            // Close the input stream and return bytes
            is.close();
        }
        catch (FileNotFoundException e)
        {
            throw new UncheckedException(e);
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
        
        return bytes;
    }
    
    private byte[] getResourceNotFoundErrorPage(BrowsersResourceRequest usersRequest, FailureReasons reasonsForFailure)
    {
        String resourceNameString = usersRequest == null ? "" : ", <pre>" + usersRequest.getURIString() + "</pre>,";
        
        String reasonOriginServerFailed = reasonsForFailure.getReasonOriginServerFailed() == null ? "" : "<p>Reason origin server failed to serve resource:<br>" + reasonsForFailure.getReasonOriginServerFailed() + "</p>";
        String reasonCacheFailed = reasonsForFailure.getReasonCacheFailed() == null ? "" : "<p>Reason client cache failed to serve resource:<br>" + reasonsForFailure.getReasonCacheFailed() + "</p>";
        String reasonPeersFailed = reasonsForFailure.getReasonPeersFailed() == null ? "" : "<p>Reason peers failed to serve resource:<br>" + reasonsForFailure.getReasonPeersFailed() + "</p>";
        String reasonUnknownFailed = reasonsForFailure.getReasonUnknownFailed() == null ? "" : "<p>Other reason for failure to server resource:<br>" + reasonsForFailure.getReasonUnknownFailed() + "</p>";
                
        String output = ""
                + "<html>"
                + "<head>"
                + "<title>NoTorrent Error: Resource Could Not Be Retrieved</title>"
                + Constants.ERRORPAGE_STYLE_HEADER
                + "</head>"
                + "<body>"
                + "<p>I'm sorry, but the resource you requested" + resourceNameString + " could not be retrieved via NoTorrent.</p>"
                + reasonOriginServerFailed
                + reasonCacheFailed
                + reasonPeersFailed
                + reasonUnknownFailed
                + "</body>"
                + "</html>";
        
        return output.getBytes();
    }
    
    private void cacheRetrievedResourceAndInformTracker(byte[] retrievedResource, URI requestedUri)
    {
        ClientCache.cacheRetrievedResource(retrievedResource, requestedUri);
        informTrackerIHaveResource(requestedUri);
    }
    
    private void informTrackerIHaveResource(URI requestedUri)
    {
        try
        {
            Message iHaveResourceMessage = new MessageIHaveResource(requestedUri, InetAddress.getLocalHost(), parentClient.getClientServerPort());
            //	System.out.println(".getMyAddress():" + ((MessagePeerListRequest)peerListRequestMessage).getMyAddress().toString());
            //	System.out.println("InetAddress.getLocalHost():" + InetAddress.getLocalHost().toString());
            
            String trackerName = parentClient.getTrackerAddress().toString() + ":" + parentClient.getTrackerPort();
            
            System.out.println("Message to be sent to tracker (" + trackerName + "):\n" + iHaveResourceMessage);
            
            // connect to tracker
            Socket socket = new Socket(parentClient.getTrackerAddress(), parentClient.getTrackerPort());
            System.out.println("Connected to tracker...sending iHaveResourceMessage message");
            
            BufferedOutputStream outputStream = new BufferedOutputStream(socket.getOutputStream());
            
            // send the encoded message to the tracker
            outputStream.write(iHaveResourceMessage.encode());
            outputStream.flush();
            
            // closing of output stream indicates end of message
            //	outputStream.close();
            socket.close();
            
            // no need to wait for a response, because there will be none.
        }
        catch (ConnectException e)
        {
            System.err.println("Could not inform tracker that I have resource \"" + requestedUri.toString() + ".\"  Tracker appears to be down.");
        }
        catch (UnknownHostException e)
        {
            throw new UncheckedException(e);
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
    }
    
    private void informTrackerThatPeerFailedToServe(URI requestedUri, Peer peerToAsk)
    {
        try
        {
            Message peerFailedToServeMessage = new MessagePeerFailedToServe(requestedUri, peerToAsk.getAddress(), peerToAsk.getPort());
            //	System.out.println(".getMyAddress():" + ((MessagePeerListRequest)peerListRequestMessage).getMyAddress().toString());
            //	System.out.println("InetAddress.getLocalHost():" + InetAddress.getLocalHost().toString());
            
            System.out.println("Message to be sent to tracker:\n" + peerFailedToServeMessage);
            
            // connect to tracker
            Socket socket = new Socket(parentClient.getTrackerAddress(), parentClient.getTrackerPort());
            System.out.println("Connected to tracker...sending peerFailedToServeMessage message");
            
            BufferedOutputStream outputStream = new BufferedOutputStream(socket.getOutputStream());
            
            // send the encoded message to the tracker
            outputStream.write(peerFailedToServeMessage.encode());
            outputStream.flush();
            
            // closing of output stream indicates end of message
            //	outputStream.close();
            socket.close();
            
            // no need to wait for a response, because there will be none.
        }
        catch (UnknownHostException e)
        {
            throw new UncheckedException(e);
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
    }
    
    public static byte[] prependStringToResource(byte[] retrievedResource, String stringToPrepend)
    {
        //////////////////////////////////////////////////////////////////////
        // manipulate the content if I'm feeling funny.
//            serverResponse = serverResponse.replaceAll("I ", "you ");
//	    serverResponse = serverResponse.replaceFirst("(?i)<body([^>]*)>", "<body$1><p>This page was served via PeDiCache.</p><hr>");
        
        
        System.out.println("in manipulateResource");
        if (retrievedResource == null)
        {
            return null;
        }
        
        String retrievedResourceString = new String(retrievedResource);
        
        // if a body tag is found, prepend a "served via NoTorrent" message
        // the s flag means single-line or DOTALL mode, which means the . will also match line terminators.
        // the i flag means case-insensitive
        if (retrievedResourceString.matches("(?si).*<body.*>.*"))
        {
            System.out.println("matches body regex");
            String modifiedResourceString = retrievedResourceString.replaceFirst("(?i)<body([^>]*)>", "<body$1><p" + Constants.STYLE_ATTRIBUTE + ">" + stringToPrepend + "</p><hr>");
            return modifiedResourceString.getBytes();
        }
        else
        {
            System.out.println("doesn't match body regex");
        }
        
        System.out.println("return unmodified resource");
        // if no body tag was found (meaning it probably wasn't an html document), return the unmodified resource
        return retrievedResource;
    }
}
