/*
 * Client.java
 *
 */

package notorrent.client;

import java.io.IOException;
import java.net.BindException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.UnknownHostException;
import notorrent.util.MethodNotImplementedException;
import notorrent.util.UncheckedException;

/**
 *
 * @author hrv2101
 */
public class Client
{
    private CommandLineArgs commandLineArgs;
    
    private int clientServerPort;
    private int clientProxyPort;
    
    private ServerSocket clientServerSocket = null;
    private ServerSocket clientProxySocket = null;
    
    /** Creates a new instance of Client */
    public Client()
    {
        commandLineArgs = new CommandLineArgs();
    }
    
    public static void main(String[] args)
    {
        Client client = new Client();
        client.run(args);
    }
    
    private void run(String[] args)
    {
        parseCommandLineArgs(args);
        initializeCache();
        spawnClientProxy();
        spawnClientServer();
    }
    
    /**
     * sets client settings based on command line arguments.
     */
    private void parseCommandLineArgs(String[] args)
    {
        commandLineArgs.parseCommandLineArgs(args);
    }
    
    private void initializeCache()
    {
        ClientCache.initialize();
    }
    
    /**
     * this should actually keep searching until an free port is found.  for now,
     * we only check the one specified port.
     */
    private void startClientServerSocket()
    {
        clientServerPort = commandLineArgs.getFirstClientServerPortToTry();
        
        for (int i = 0; i < commandLineArgs.getMaxNumClientServerPortsToTry(); i++)
        {
            try
            {
                System.out.println("ClientServer trying to listen on port " + clientServerPort + ".");
                
                //	    clientServerSocket = new ServerSocket(commandLineArgs.getPortToListenOn());
                clientServerSocket = new ServerSocket(clientServerPort);
                //	    int clientServerPort = clientServerSocket.getLocalPort();
                //	    String serverAddress = serverSocket.getInetAddress().getHostAddress();
                String clientServerAddress = InetAddress.getLocalHost().getHostAddress();
                System.out.println("Client will listen for connections from other peers on port " + clientServerPort + " at address " + clientServerAddress + ".");
                
                // we started the socket without throwing an exception.  the socket is bound, so we return.
                return;
            }
            catch (BindException e)
            {
                System.err.println("Could not start ClientServer on port " + clientServerPort + ".  It is already in use.");
                
                // we'll try to connect on the next port.
                clientServerPort++;
            }
            catch (IOException e)
            {
                throw new UncheckedException(e);
            }
        }
        
        System.err.println("Could not start ClientServer.  NoTorrent is terminating.");
        System.exit(1);
    }
    
    private void startClientProxySocket()
    {
        try
        {
//	    clientServerSocket = new ServerSocket(commandLineArgs.getPortToListenOn());
            clientProxySocket = new ServerSocket(commandLineArgs.getClientProxyPort());
            int clientProxyPort = clientProxySocket.getLocalPort();
//	    String serverAddress = serverSocket.getInetAddress().getHostAddress();
            String clientProxyAddress = InetAddress.getLocalHost().getHostAddress();
            System.out.println("Client proxy will listen for connections from the browser on port " + clientProxyPort + " at address " + clientProxyAddress + ".");
        }
        catch (BindException e)
        {
            System.err.println("Could not start ClientProxy.  Port " + commandLineArgs.getClientProxyPort() + " is already in use.  NoTorrent is terminating.");
            System.exit(1);
        }
        catch (IOException e)
        {
            throw new UncheckedException(e);
        }
    }
    
    /**
     * we handle peer connections by creating a pool of threads.
     * each thread is responsible for listening for connections,
     * handling any connection that comes in, and then listening for
     * connections again.
     */
    private void spawnClientServer()
    {
        startClientServerSocket();
        
        // dispatch numThreads HandlerThreads
//	for (int i = 0; i < commandLineArgs.getNumThreads(); i++)
        for (int i = 0; i < commandLineArgs.getNumClientServerThreads(); i++)
        {
            Runnable clientServerThread = new ClientServerThread(this);
            new Thread(clientServerThread).start();
        }
    }
    
    private void spawnClientProxy()
    {
        startClientProxySocket();
        
        for (int i = 0; i < commandLineArgs.getNumClientProxyThreads(); i++)
        {
            Runnable clientProxyThread = new ClientProxyThread(this);
            new Thread(clientProxyThread).start();
        }
    }
    
    public ServerSocket getClientServerSocket()
    {
        return clientServerSocket;
    }
    
    public int getClientServerPort()
    {
        return clientServerPort;
    }
    
    public ServerSocket getClientProxySocket()
    {
        return clientProxySocket;
    }
    
    public InetAddress getTrackerAddress()
    {
        InetAddress trackerAddress = null;
        
        try
        {
            trackerAddress = InetAddress.getByName(commandLineArgs.getTrackerAddress());
        }
        catch (UnknownHostException e)
        {
            throw new UncheckedException(e);
        }
        
        return trackerAddress;
    }
    
    public int getTrackerPort()
    {
        return commandLineArgs.getTrackerPort();
    }
}
